/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx__error.c,v 1.24 2006/07/26 01:28:22 loic Exp $";

#include "mx_auto_config.h"
#include "myriexpress.h"
#include "mx__lib_types.h"
#include "mx__error.h"
#include "mx__lib.h"
#include "mx__debug_dump.h"
#include "mx__endpoint.h"

static char this_hostname[100] =  "";

int mx__error_set_verbose(int flag)
{
  int old = mx__opt.verbose_errors;
#if MX_RUNTIME_OPT
  mx__opt.verbose_errors = flag;
#endif
  return old;
}

static mx_return_t 
mx__errors_return(char *msg, mx_return_t ret)
{
  return ret;
}

#ifndef MX_KERNEL
static mx_return_t 
mx__errors_no_init(char *msg, mx_return_t ret)
{
  mx_printf("MX:error encountered(%s:%s) before mx_init was called\n", 
	   msg, mx_strerror(ret));
  mx_printf("MX:Aborting\n");
  exit(2);
  return ret;
}

static mx_return_t 
mx__errors_are_fatal(char *msg, mx_return_t ret)
{
  mx_printf("MX:Aborting\n");
  mx__abort();
  return ret;
}
#endif

MX_VAR(const mx_error_handler_t) MX_ERRORS_RETURN = mx__errors_return;

#ifndef MX_KERNEL
MX_VAR(const mx_error_handler_t) MX_ERRORS_ARE_FATAL = mx__errors_are_fatal;
static mx_error_handler_t mx__error_handler = mx__errors_no_init;
#else
static mx_error_handler_t mx__error_handler = mx__errors_return;
#endif

static int
mx__verbose_errors(void)
{
  int ret = mx__opt.verbose_errors;
#ifndef MX_KERNEL
  ret = ret || (mx__error_handler == mx__errors_are_fatal);
#endif
  return ret;
}
int mx__error_req(struct mx_endpoint *ep, char *msg, 
		  union mx_request *r, int ret, ...)
{
  char buf[200];
  va_list va;
  mx_error_handler_t ep_handler;

  if (ret == MX_SUCCESS)
    return ret;
  va_start(va, ret);
  vsprintf(buf,msg, va);
  va_end(va);
  ep_handler = ep->error_handler;
  if (ep_handler) {
    ep_handler(buf, ret);
    return ret;
  }
  if (mx__verbose_errors()) {
    mx_printf("MX:%s:%s:req status %d:%s\n", this_hostname, buf, ret, mx_strstatus(ret));
    if (r) {
      mx__dump_request(ep, r);
    }
  }
  return mx__error_handler(buf, ret);
}

int mx__error_noep(char *msg, int ret, ...)
{
  char buf[200];
  va_list va;

  if (ret == MX_SUCCESS)
    return ret;
  if (ret == MX_HOST_NOT_FOUND) {
    /* should we deal with usual errors in a special manner?  maybe
       two different level of verbosity for "manifest program errors"
       as opposed to "usual errors" */
  }
  va_start(va, ret);
  vsprintf(buf,msg, va);
  va_end(va);
  if (mx__verbose_errors()) {
    mx_printf("MX:%s:%s:error %d(errno=%d):%s\n", this_hostname, buf, ret, mx_errno, mx_strerror(ret));
    if (ret == MX_BAD_BAD_BAD) {
      mx_printf("\tlast syscall error=%d:%s\n", mx_errno, mx_strerrno);
    }
  }
  return mx__error_handler(buf, ret);
}

int mx__error(struct mx_endpoint *ep, char *msg, int ret, ...)
{
  char buf[200];
  va_list va;
  
  if (ret == MX_SUCCESS)
    return ret;
  va_start(va, ret);
  vsprintf(buf,msg, va);
  va_end(va);
  if (ep->error_handler) {
    ep->error_handler(buf, ret);
    return ret;
  }
  if (mx__verbose_errors()) {
    mx_printf("MX:%s:%s:error %d(errno=%d):%s\n", this_hostname, buf, ret, mx_errno, mx_strerror(ret));
  }
  return mx__error_handler(buf, ret);
}


#ifndef MX_KERNEL
MX_FUNC(mx_error_handler_t)
mx_set_error_handler(mx_error_handler_t func)
{
  mx_error_handler_t old = mx__error_handler;
  mx__error_handler = func;
  return old;
}
#endif


void mx__error_init(void)
{
#ifndef MX_KERNEL
#if !MX_OS_WINNT
  struct utsname uts;
  if (mx__error_handler == mx__errors_no_init) {
    mx__error_handler = mx__opt.errors_are_fatal ?
      mx__errors_are_fatal : mx__errors_return;
  }
  uname(&uts);
  strcpy(this_hostname,uts.nodename);
#else
  char buff[MAX_COMPUTERNAME_LENGTH+1];
  DWORD size = MAX_COMPUTERNAME_LENGTH+1;
  if (mx__error_handler == mx__errors_no_init) {
    mx__error_handler = mx__opt.errors_are_fatal ?
      mx__errors_are_fatal : mx__errors_return;
  }
  if (GetComputerName(buff, &size) != 0) {
    strcpy(this_hostname, buff);
  }
#endif
#endif
}

#ifndef MX_KERNEL
void mx__abort()
{
  if (mx__opt.verbose && Mx_endpoints) {
    int full = mx__opt.verbose > 1;
    mx__opt.verbose = 0; /* avoid reentrancy */
    mx__dump_endpoint(Mx_endpoints, full);
  }
  if (mx__opt.abort_sleeps) {
    while (1) {
      mx_sleep(1);
    }
  } else if (mx__opt.coredump) {
    abort();
  }
  exit (1);
}
#endif

#ifndef MX_KERNEL
void mx_printf(const char *fmt, ...)
{
  static FILE * err_file;
  va_list va;
  char buf[1000];

  if (!err_file) {
    char *oname = getenv("MX_DEBUG_FILE");
    if (oname)
      err_file = fopen(oname, "a");
  }
  if (!err_file)
    err_file = stderr;
  va_start(va,fmt);
  vsprintf(buf, fmt, va);
  fprintf(err_file, "%s", buf);
  fflush(err_file);
  return;
}


void 
mx_assertion_failed (const char *assertion, int line, const char *file)
{
  mx_printf("MX: assertion: <<%s>>  failed at line %d, file %s\n",
	    assertion, line, file);
  mx__abort();
}

#endif
